#ifndef __SEP_DRIVER_CONFIG_H__
#define __SEP_DRIVER_CONFIG_H__


/*--------------------------------------
  DRIVER CONFIGURATION FLAGS
  -------------------------------------*/

/* if flag is on , then the driver is running in polling and not interrupt mode */
#define SEP_DRIVER_POLLING_MODE                         1

/* flag which defines if the shared area address should be reconfiged (send to SEP anew) during init of the driver */
#define SEP_DRIVER_RECONFIG_MESSAGE_AREA                0

/* the mode for running on the ARM1172 Evaluation platform (flag is 1) */
#define SEP_DRIVER_ARM_DEBUG_MODE                       0

/* RAR lock mode - cahce, resident and ext cache must be copied to external RAR and locked by driver in the same API */
#define SEP_DRIVER_LOCK_RAR_MODE                        0

/* flag which defines if the SEP driver should config the OTP (flag is 1), or not (flag is 0) */
#define SEP_DRIVER_CONFIG_OTP                           1

/*-------------------------------------------
  INTERNAL DATA CONFIGURATION
  -------------------------------------------*/

/* flag for the input array */
#define SEP_DRIVER_IN_FLAG                              0

/* flag for output array */
#define SEP_DRIVER_OUT_FLAG                             1

/* maximum number of entries in one LLI tables */
#define SEP_DRIVER_ENTRIES_PER_TABLE_IN_SEP             8


/*-----------------------------------------------------------------------------------
  SHARED AREA  memory total size is 36K
  it is divided is following:                                      
  
    SHARED_MESSAGE_AREA                     8K             }
                                                            } 
    STATIC_POOL_AREA                        4K               }   MAPPED AREA ( 24 K)  
                                                            }
    DATA_POOL_AREA                          12K            }
    
    SYNCHRONIC_DMA_TABLES_AREA              5K
 
    FLOW_DMA_TABLES_AREA                    4K
    
    SYSTEM_MEMORY_AREA                      3k
    
    SYSTEM_MEMORY total size is 3k
       it is divided as following:
       
       TIME_MEMORY_AREA                     8B
       CURRENT_CALLER_ID_MEMORY             70B
    
--------------------------------------------------------------------------------------*/           



    


/* 
  the maximum length of the message - the rest of the message shared
  area will be dedicated to the dma lli tables 
*/
#define SEP_DRIVER_MAX_MESSAGE_SIZE_IN_BYTES                  (8 * 1024) 

/* the size of the message shared area in pages */
#define SEP_DRIVER_MESSAGE_SHARED_AREA_SIZE_IN_BYTES          (8 * 1024)

/* the size of the data pool static area in pages */
#define SEP_DRIVER_STATIC_AREA_SIZE_IN_BYTES                  (4 * 1024)

/* the size of the data pool shared area size in pages */
#define SEP_DRIVER_DATA_POOL_SHARED_AREA_SIZE_IN_BYTES        (12 * 1024)

/* the size of the message shared area in pages */
#define SEP_DRIVER_SYNCHRONIC_DMA_TABLES_AREA_SIZE_IN_BYTES   (1024 * 5)


/* the size of the data pool shared area size in pages */
#define SEP_DRIVER_FLOW_DMA_TABLES_AREA_SIZE_IN_BYTES         (1024 * 4)

/* system data (time, caller id etc') pool */
#define SEP_DRIVER_SYSTEM_DATA_MEMORY_SIZE_IN_BYTES           (1024 * 3)


/* area size that is mapped  - we map the MESSAGE AREA, STATIC POOL and DATA POOL areas. area must be module 4k */
#define SEP_DRIVER_MMMAP_AREA_SIZE                            (1024 * 24)


/*-----------------------------------------------
   offsets of the areas starting from the shared area start address 
*/

/* message area offset */
#define SEP_DRIVER_MESSAGE_AREA_OFFSET_IN_BYTES                 0

/* static pool area offset */
#define SEP_DRIVER_STATIC_AREA_OFFSET_IN_BYTES                  (SEP_DRIVER_MESSAGE_SHARED_AREA_SIZE_IN_BYTES) 

/* data pool area offset */
#define SEP_DRIVER_DATA_POOL_AREA_OFFSET_IN_BYTES               (SEP_DRIVER_STATIC_AREA_OFFSET_IN_BYTES + SEP_DRIVER_STATIC_AREA_SIZE_IN_BYTES)

/* synhronic dma tables area offset */
#define SEP_DRIVER_SYNCHRONIC_DMA_TABLES_AREA_OFFSET_IN_BYTES   (SEP_DRIVER_DATA_POOL_AREA_OFFSET_IN_BYTES + SEP_DRIVER_DATA_POOL_SHARED_AREA_SIZE_IN_BYTES)

/* sep driver flow dma tables area offset */
#define SEP_DRIVER_FLOW_DMA_TABLES_AREA_OFFSET_IN_BYTES         (SEP_DRIVER_SYNCHRONIC_DMA_TABLES_AREA_OFFSET_IN_BYTES + SEP_DRIVER_SYNCHRONIC_DMA_TABLES_AREA_SIZE_IN_BYTES)

/* system memory offset in bytes */
#define SEP_DRIVER_SYSTEM_DATA_MEMORY_OFFSET_IN_BYTES           (SEP_DRIVER_FLOW_DMA_TABLES_AREA_OFFSET_IN_BYTES + SEP_DRIVER_FLOW_DMA_TABLES_AREA_SIZE_IN_BYTES)


/* offset of the time area */
#define SEP_DRIVER_SYSTEM_TIME_MEMORY_OFFSET_IN_BYTES           (SEP_DRIVER_SYSTEM_DATA_MEMORY_OFFSET_IN_BYTES)

/* offset of the caller id area */
#define SEP_DRIVER_SYSTEM_CALLER_ID_MEMORY_OFFSET_IN_BYTES      (SEP_DRIVER_SYSTEM_TIME_MEMORY_OFFSET_IN_BYTES + 8)


/* start physical address of the SEP registers memory in HOST */
#define SEP_IO_MEM_REGION_START_ADDRESS                       0x80000000

/* size of the SEP registers memory region  in HOST (for now 100 registers) */
#define SEP_IO_MEM_REGION_SIZE                                (2 * 0x100000)

/* define the number of IRQ for SEP interrupts */
#define SEP_DIRVER_IRQ_NUM                                    1

/* maximum number of add buffers */
#define SEP_MAX_NUM_ADD_BUFFERS                               100

/* number of flows */
#define SEP_DRIVER_NUM_FLOWS                                  4

/* maximum number of entries in flow table */
#define SEP_DRIVER_MAX_FLOW_NUM_ENTRIES_IN_TABLE              25

/* offset of the num entries in the block length entry of the LLI */
#define SEP_NUM_ENTRIES_OFFSET_IN_BITS                        24

/* offset of the interrupt flag in the block length entry of the LLI */
#define SEP_INT_FLAG_OFFSET_IN_BITS                           31

/* mask for extracting data size from LLI */
#define SEP_TABLE_DATA_SIZE_MASK                              0xFFFFFF

/* mask for entries after being shifted left */
#define SEP_NUM_ENTRIES_MASK                                  0x7F

/* default flow id */
#define SEP_FREE_FLOW_ID                                      0xFFFFFFFF

/* temp flow id used during cretiong of new flow until receiving real flow id from sep */
#define SEP_TEMP_FLOW_ID                                      (SEP_DRIVER_NUM_FLOWS + 1)

/* maximum add buffers message length in bytes */
#define SEP_MAX_ADD_MESSAGE_LENGTH_IN_BYTES                   (7 * 4)

/* maximum number of concurrent virtual buffers */ 
#define SEP_MAX_VIRT_BUFFERS_CONCURRENT                       100

/* the token that defines the start of time address */
#define SEP_TIME_VAL_TOKEN                                    0x12345678

/* size of the caller id hash (sha2) */
#define SEP_CALLER_ID_HASH_SIZE_IN_BYTES                      32

/* maximum number of entries in the caller id table */
#define SEP_CALLER_ID_TABLE_NUM_ENTRIES                       20

/* ioctl error that should be returned when trying to realloc the cache/resident second time */
#define SEP_ALREADY_INITIALIZED_ERR                           12

/* DEBUG LEVEL MASKS */
#define SEP_DEBUG_LEVEL_BASIC       0x1

#define SEP_DEBUG_LEVEL_REGISTERS   0x2

#define SEP_DEBUG_LEVEL_EXTENDED    0x4


/* FUNCTIONAL MACROS */

/* debug macro without paramaters */
#define DEBUG_PRINT_0(DEBUG_LEVEL , info) \
do \
{ \
  if(DEBUG_LEVEL & sepDebug) \
  { \
    printk(KERN_WARNING info); \
  } \
}while(0)

/* debug macro with 1 paramater */
#define DEBUG_PRINT_1(DEBUG_LEVEL , info , param1) \
do \
{ \
  if(DEBUG_LEVEL & sepDebug) \
  { \
    printk(KERN_WARNING info , param1); \
  } \
}while(0)

/* debug macro with 2 paramaters */
#define DEBUG_PRINT_2(DEBUG_LEVEL , info , param1, param2) \
do \
{ \
  if(DEBUG_LEVEL & sepDebug) \
  { \
    printk(KERN_WARNING info , param1, param2); \
  } \
}while(0)

/* debug macro with 3 paramaters */
#define DEBUG_PRINT_3(DEBUG_LEVEL , info , param1, param2 , param3) \
do \
{ \
  if(DEBUG_LEVEL & sepDebug) \
  { \
    printk(KERN_WARNING info , param1, param2 , param3); \
  } \
}while(0)

/* debug macro with 4 paramaters */
#define DEBUG_PRINT_4(DEBUG_LEVEL , info , param1, param2 , param3 , param4) \
do \
{ \
  if(DEBUG_LEVEL & sepDebug) \
  { \
    printk(KERN_WARNING info , param1, param2 , param3 , param4); \
  } \
}while(0)

#if 0
/* write register macro with option for debug print */
#define SEP_WRITE_REGISTER(address , value) \
do \
{ \
  if(sepDebug & SEP_DEBUG_LEVEL_REGISTERS) \
  { \
    printk(KERN_WARNING "Write Register: address %lu value %lu\n" , (unsigned long)(address) , (unsigned long)(value)); \
  } \
  writel( (value) , (void*)(address) ); \
}while(0)

/* read register macro with option for debug print */  
#define SEP_READ_REGISTER(address , value) \
do \
{ \
  (value) = readl( (void*) (address) ); \
  if(sepDebug & SEP_DEBUG_LEVEL_REGISTERS) \
  { \
    printk(KERN_WARNING "Read Register: address %lu value %lu\n" , (address) , (value)); \
  } \
}while(0)
#else

#if 1

#define SEP_WRITE_REGISTER(address , value)  writel( (value) , (void*)(address))
#define SEP_READ_REGISTER(address , value)   (value) = readl( (void*) (address))
#endif

#endif

#if 0
#define SEP_WRITE_ROM(address , value)  writel( (value) , (void*)(address))

#define SEP_WRITE_REGISTER(address , value) \
do \
{ \
  volatile unsigned long i; \
  for(i = 0; i < 1000;i++); \
  writel( (value) , (void*)(address)); \
}while(0)


#define SEP_READ_REGISTER(address , value) \
do \
{ \
  volatile unsigned long i; \
  for(i = 0; i < 1000;i++); \
  (value) = readl( (void*) (address)); \
}while(0)

#endif

/* wait for SRAM write complete(indirect write */
#define SEP_WAIT_SRAM_WRITE_COMPLETE() \
do \
{ \
  unsigned long  reg_val; \
  do \
  { \
    SEP_READ_REGISTER(g_sep_reg_base_address + HW_SRAM_DATA_READY_REG_ADDR, (reg_val) ); \
  }while(!(reg_val & 0x1)); \
}while(0)



#endif
